/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import { generateUuid } from "@kie-tools/boxed-expression-component/dist/api";
import { ExternalDmnsIndex, ExternalModelsIndex, ExternalPmmlsIndex } from "../DmnEditor";
import { computeDiagramData } from "../store/computed/computeDiagramData";
import { State } from "../store/Store";
import { MIN_NODE_SIZES } from "../diagram/nodes/DefaultSizes";
import { getNodeTypeFromDmnObject } from "../diagram/maths/DmnMaths";
import { DMN15__tDefinitions } from "@kie-tools/dmn-marshaller/dist/schemas/dmn-1_5/ts-gen/types";
import { DmnLatestModel } from "@kie-tools/dmn-marshaller";
import { Normalized } from "@kie-tools/dmn-marshaller/dist/normalization/normalize";
import { parseXmlHref } from "@kie-tools/dmn-marshaller/dist/xml/xmlHrefs";
import { computeIndexedDrd } from "../store/computed/computeIndexes";
import { getDefaultDrdName } from "../mutations/addOrGetDrd";
import { addShape } from "../mutations/addShape";
import { addEdge } from "../mutations/addEdge";
import { EdgeType, NodeType } from "../diagram/connections/graphStructure";
import { PositionalNodeHandleId } from "../diagram/connections/PositionalNodeHandles";

export async function autoGenerateDrd(args: {
  model: State["dmn"]["model"];
  diagram: State["diagram"];
  externalModelsByNamespace: ExternalModelsIndex | undefined;
  externalModelTypesByNamespace: {
    dmns: ExternalDmnsIndex;
    pmmls: ExternalPmmlsIndex;
  };
}) {
  // Create DRD
  args.model.definitions["dmndi:DMNDI"] = {
    ...args.model.definitions["dmndi:DMNDI"],
    "dmndi:DMNDiagram": [
      {
        "@_id": generateUuid(),
        "@_name": getDefaultDrdName({ drdIndex: 0 }),
        "@_useAlternativeInputDataShape": false,
        "dmndi:DMNDiagramElement": [],
        "di:extension": { "kie:ComponentsWidthsExtension": { "kie:ComponentWidths": [{}] } },
      },
    ],
  };

  // 1. Add shapes from current DRG
  args.model.definitions.drgElement?.forEach((drgElement) => {
    const nodeType = getNodeTypeFromDmnObject(drgElement) ?? "node_unknown";
    const minNodeSize = MIN_NODE_SIZES[nodeType]({
      snapGrid: {
        isEnabled: true,
        x: 20,
        y: 20,
      },
      isAlternativeInputDataShape: false,
    });

    addShape({
      definitions: args.model.definitions,
      drdIndex: 0,
      nodeType: nodeType,
      shape: {
        "@_id": generateUuid(),
        "@_dmnElementRef": drgElement["@_id"]!,
        "dc:Bounds": {
          "@_x": 0,
          "@_y": 0,
          ...minNodeSize,
        },
      },
    });
  });

  // 2. Add shapes from external models;
  const definedNamespaces = new Map(
    Object.keys(args.model.definitions)
      .filter((keys: keyof Normalized<DMN15__tDefinitions>) => String(keys).startsWith("@_xmlns:"))
      .map((xmlnsKey: keyof Normalized<DMN15__tDefinitions>) => [
        args.model.definitions[xmlnsKey],
        xmlnsKey.split("@_xmlns:")[1],
      ])
  );

  const updateIndexedDrdWithNodes = computeIndexedDrd(args.model.definitions["@_namespace"], args.model.definitions, 0);
  const { nodesById: updatedNodesByIdWithNodes, drgEdges: updatedDrgEdgesWithNodes } = computeDiagramData(
    args.diagram,
    args.model.definitions,
    args.externalModelTypesByNamespace,
    updateIndexedDrdWithNodes,
    false
  );

  // Search on edges for any node that isn't on the nodesByIdMap;
  // Only external nodes should be added to the externalNodesHref;
  const externalNodesHref = updatedDrgEdgesWithNodes.reduce((acc, drgEdge) => {
    if (!updatedNodesByIdWithNodes.has(drgEdge.sourceId)) {
      acc.add(drgEdge.sourceId);
    }
    if (!updatedNodesByIdWithNodes.has(drgEdge.targetId)) {
      acc.add(drgEdge.targetId);
    }
    return acc;
  }, new Set<string>());

  // Add external shapes
  externalNodesHref.forEach((href) => {
    const { namespace, id } = parseXmlHref(href);
    if (namespace) {
      const externalModel = args.externalModelsByNamespace?.[namespace];
      if (externalModel && (externalModel.model as Normalized<DmnLatestModel>).definitions) {
        const drgElements = (externalModel.model as Normalized<DmnLatestModel>).definitions.drgElement;
        const drgElement = drgElements?.filter((drgElement) => drgElement["@_id"] === id);

        const nodeType = getNodeTypeFromDmnObject(drgElement![0]) ?? "node_unknown";
        const minNodeSize = MIN_NODE_SIZES[nodeType]({
          snapGrid: {
            isEnabled: true,
            x: 20,
            y: 20,
          },
          isAlternativeInputDataShape: false,
        });

        addShape({
          definitions: args.model.definitions,
          drdIndex: 0,
          nodeType: nodeType,
          shape: {
            "@_id": generateUuid(),
            "@_dmnElementRef": `${definedNamespaces.get(namespace)}:${id}`,
            "dc:Bounds": {
              "@_x": 0,
              "@_y": 0,
              ...minNodeSize,
            },
          },
        });
      }
    }
  });

  // 3. Add edges
  const updatedIndexedDrdWithExternalNodes = computeIndexedDrd(
    args.model.definitions["@_namespace"],
    args.model.definitions,
    0
  );
  const {
    nodesById: updatedNodesByIdWithExternalNodes,
    edgesById: updatedEdgesByIdWithExternalNodes,
    drgEdges: updatedDrgEdgesWithExternalNodes,
  } = computeDiagramData(
    args.diagram,
    args.model.definitions,
    args.externalModelTypesByNamespace,
    updatedIndexedDrdWithExternalNodes,
    false
  );

  for (const drgEdge of updatedDrgEdgesWithExternalNodes) {
    const edge = updatedEdgesByIdWithExternalNodes.get(drgEdge.id);
    const sourceNode = updatedNodesByIdWithExternalNodes.get(drgEdge.sourceId);
    const targetNode = updatedNodesByIdWithExternalNodes.get(drgEdge.targetId);

    // Avoid missing nodes. Possible cause is an external model which couldn't be found.
    if (!edge || !sourceNode || !targetNode) {
      continue;
    }

    addEdge({
      definitions: args.model.definitions,
      drdIndex: 0,
      keepWaypoints: false,
      edge: {
        autoPositionedEdgeMarker: undefined,
        type: edge.type as EdgeType,
        targetHandle: PositionalNodeHandleId.Bottom,
        sourceHandle: PositionalNodeHandleId.Top,
      },
      sourceNode: {
        type: sourceNode.type as NodeType,
        href: sourceNode.id,
        data: sourceNode.data,
        bounds: sourceNode.data.shape["dc:Bounds"]!,
        shapeId: sourceNode.data.shape["@_id"],
      },
      targetNode: {
        type: targetNode.type as NodeType,
        href: targetNode.id,
        data: targetNode.data,
        bounds: targetNode.data.shape["dc:Bounds"]!,
        index: targetNode.data.index,
        shapeId: targetNode.data.shape["@_id"],
      },
      externalModelsByNamespace: args.externalModelsByNamespace,
    });
  }
}
