/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.ftpserver.ftpletcontainer.impl;

import java.io.IOException;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.ftpserver.ftplet.FtpException;
import org.apache.ftpserver.ftplet.FtpReply;
import org.apache.ftpserver.ftplet.FtpRequest;
import org.apache.ftpserver.ftplet.FtpSession;
import org.apache.ftpserver.ftplet.Ftplet;
import org.apache.ftpserver.ftplet.FtpletContext;
import org.apache.ftpserver.ftplet.FtpletResult;
import org.apache.ftpserver.ftpletcontainer.FtpletContainer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <strong>Internal class, do not use directly.</strong>
 * <p>
 * This ftplet calls other ftplet methods and returns appropriate return value.
 * <p>
 * <strong><strong>Internal class, do not use directly.</strong></strong>
 *
 * @author <a href="http://mina.apache.org">Apache MINA Project</a>
 */
public class DefaultFtpletContainer implements FtpletContainer {
    /** The class logger */
    private final Logger LOG = LoggerFactory.getLogger(DefaultFtpletContainer.class);

    /** The contained FtpLets */
    private final Map<String, Ftplet> ftplets;

    /**
     * Create a DefaultFtpletContainer instance
     */
    public DefaultFtpletContainer() {
        this(new ConcurrentHashMap<>());
    }

    /**
     * Create a DefaultFtpletContainer instance
     *
     * @param ftplets The set of FtpLets to store
     */
    public DefaultFtpletContainer(Map<String, Ftplet> ftplets) {
        this.ftplets = ftplets;
    }

    /**
     * {@inheritDoc}
     */
    public synchronized Ftplet getFtplet(String name) {
        if (name == null) {
            return null;
        }

        return ftplets.get(name);
    }

    /**
     * Initialize teh container
     *
     * @param ftpletContext The context to use
     * @throws FtpException If the initialization failed
     */
    public synchronized void init(FtpletContext ftpletContext) throws FtpException {
        for (Entry<String, Ftplet> entry : ftplets.entrySet()) {
            entry.getValue().init(ftpletContext);
        }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized Map<String, Ftplet> getFtplets() {
        return ftplets;
    }

    /**
     * {@inheritDoc}
     */
    public void destroy() {
        for (Entry<String, Ftplet> entry : ftplets.entrySet()) {
            try {
                entry.getValue().destroy();
            } catch (Exception ex) {
                LOG.error(entry.getKey() + " :: FtpletHandler.destroy()", ex);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public FtpletResult onConnect(FtpSession session) throws FtpException, IOException {
        FtpletResult retVal = FtpletResult.DEFAULT;

        for (Entry<String, Ftplet> entry : ftplets.entrySet()) {
            retVal = entry.getValue().onConnect(session);

            if (retVal == null) {
                retVal = FtpletResult.DEFAULT;
            }

            // proceed only if the return value is FtpletResult.DEFAULT
            if (retVal != FtpletResult.DEFAULT) {
                break;
            }
        }

        return retVal;
    }

    /**
     * {@inheritDoc}
     */
    public FtpletResult onDisconnect(FtpSession session) throws FtpException, IOException {
        FtpletResult retVal = FtpletResult.DEFAULT;

        for (Entry<String, Ftplet> entry : ftplets.entrySet()) {
            retVal = entry.getValue().onDisconnect(session);

            if (retVal == null) {
                retVal = FtpletResult.DEFAULT;
            }

            // proceed only if the return value is FtpletResult.DEFAULT
            if (retVal != FtpletResult.DEFAULT) {
                break;
            }
        }

        return retVal;
    }

    /**
     * {@inheritDoc}
     */
    public FtpletResult afterCommand(FtpSession session, FtpRequest request, FtpReply reply)
            throws FtpException, IOException {
        FtpletResult retVal = FtpletResult.DEFAULT;

        for (Entry<String, Ftplet> entry : ftplets.entrySet()) {
            retVal = entry.getValue().afterCommand(session, request, reply);

            if (retVal == null) {
                retVal = FtpletResult.DEFAULT;
            }

            // proceed only if the return value is FtpletResult.DEFAULT
            if (retVal != FtpletResult.DEFAULT) {
                break;
            }
        }

        return retVal;
    }

    /**
     * {@inheritDoc}
     */
    public FtpletResult beforeCommand(FtpSession session, FtpRequest request) throws FtpException, IOException {
        FtpletResult retVal = FtpletResult.DEFAULT;

        for (Entry<String, Ftplet> entry : ftplets.entrySet()) {
            retVal = entry.getValue().beforeCommand(session, request);

            if (retVal == null) {
                retVal = FtpletResult.DEFAULT;
            }

            // proceed only if the return value is FtpletResult.DEFAULT
            if (retVal != FtpletResult.DEFAULT) {
                break;
            }
        }

        return retVal;
    }
}
