/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.websvc.rest.client;

import java.awt.Image;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JPanel;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.border.EtchedBorder;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.modules.websvc.rest.model.api.RestServiceDescription;
import org.netbeans.spi.project.ui.LogicalViewProvider;
import org.openide.DialogDescriptor;
import org.openide.explorer.ExplorerManager;
import org.openide.explorer.view.BeanTreeView;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.Children;
import org.openide.nodes.FilterNode;
import org.openide.nodes.Node;
import org.openide.util.NbBundle;

/**
 *
 * @author Milan Kuchtiak
 * Panel for displaying web services in projects
 * TODO: Needs to be moved to a common place since this serves both JAXWS and JAXRPC
 */
public class RESTExplorerPanel extends JPanel implements ExplorerManager.Provider, PropertyChangeListener {
    
    private DialogDescriptor descriptor;
    private ExplorerManager manager;
    private BeanTreeView treeView;
    private Node selectedResourceNode;
    private ProjectNodeFactory factory;
    
    private static final ProjectNodeFactory REST_FACTORY = new RestProjectNodeFactory();
    
    public RESTExplorerPanel() {
        this( REST_FACTORY );
    }
    
    public RESTExplorerPanel( ProjectNodeFactory factory) {
        manager = new ExplorerManager();
        selectedResourceNode = null;
        
        initComponents();
        initUserComponents();
        this.factory = factory;
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jLblTreeView = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        org.openide.awt.Mnemonics.setLocalizedText(jLblTreeView, NbBundle.getMessage(RESTExplorerPanel.class, "LBL_SelectRESTResource")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(11, 11, 0, 11);
        add(jLblTreeView, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel jLblTreeView;
    // End of variables declaration//GEN-END:variables
    
    private void initUserComponents() {
        treeView = new BeanTreeView();
        treeView.setRootVisible(false);
        treeView.setPopupAllowed(false);
        treeView.setBorder(new EtchedBorder());
        treeView.setDefaultActionAllowed(false);
        
        java.awt.GridBagConstraints gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(11, 11, 0, 11);
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(treeView, gridBagConstraints);
        jLblTreeView.setLabelFor(treeView.getViewport().getView());
        treeView.getAccessibleContext().setAccessibleName(NbBundle.getMessage(RESTExplorerPanel.class, "ACSD_RESTResourcesTreeView"));
        treeView.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(RESTExplorerPanel.class, "ACSD_RESTResourcesTreeView"));        
    }
    
    @Override
    public ExplorerManager getExplorerManager() {
        return manager;
    }
    
    @Override
    public void addNotify() {
        super.addNotify();
        manager.addPropertyChangeListener(this);
        Project[] projects = OpenProjects.getDefault().getOpenProjects();
        Children rootChildren = new Children.Array();
        AbstractNode explorerResourcesRoot = new AbstractNode(rootChildren);
        List<Node> projectNodeList = new ArrayList<Node>();
        for (Project prj : projects) {
            Node node = factory.createNode(prj);
            if ( node != null ){
                projectNodeList.add(node);
            }
        }
        Node[] projectNodes = new Node[projectNodeList.size()];
        projectNodeList.<Node>toArray(projectNodes);
        rootChildren.add(projectNodes);
        manager.setRootContext(explorerResourcesRoot);
        
        // !PW If we preselect a node, this can go away.
        descriptor.setValid(false);

        treeView.requestFocusInWindow();
    }
    
    @Override
    public void removeNotify() {
        manager.removePropertyChangeListener(this);
        super.removeNotify();
    }
    
    public void setDescriptor(DialogDescriptor descriptor) {
        this.descriptor = descriptor;
    }
    
    public Node getSelectedService() {
        return selectedResourceNode;
    }
    
    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if(evt.getSource() == manager) {
            if(ExplorerManager.PROP_SELECTED_NODES.equals(evt.getPropertyName())) {
                Node nodes[] = manager.getSelectedNodes();
                if(nodes != null && nodes.length > 0 ) {
                    Node node = nodes[0];
                    if( factory.canSelect(node)) {
                        // This is a resource node.
                        selectedResourceNode = node;
                        descriptor.setValid(true);
                    } else {
                        // This is not a service node.
                        selectedResourceNode = null;
                        descriptor.setValid(false);
                    }
                }
            }
        }
    }

    public static interface ProjectNodeFactory {
        Node createNode( Project project );
        boolean canSelect( Node node );
    }
    
    private static class RestProjectNodeFactory implements ProjectNodeFactory {

        /* (non-Javadoc)
         * @see org.netbeans.modules.websvc.rest.client.RESTExplorerPanel.ProjectNodeFactory#createNode(org.netbeans.api.project.Project)
         */
        @Override
        public Node createNode( Project project ) {
            LogicalViewProvider logicalProvider = project.getLookup().lookup(LogicalViewProvider.class);
            if (logicalProvider!=null) {
                Node rootNode = logicalProvider.createLogicalView();
                Node restResourcesNode = RESTResourcesView.
                createRESTResourcesView(project);
                if (restResourcesNode != null) {
                    return new ProjectNode(
                            new FilterNode.Children(restResourcesNode), rootNode);
                }
            }
            return null;
        }

        /* (non-Javadoc)
         * @see org.netbeans.modules.websvc.rest.client.RESTExplorerPanel.ProjectNodeFactory#canSelect(org.openide.nodes.Node)
         */
        @Override
        public boolean canSelect( Node node ) {
            return node.getLookup().lookup(RestServiceDescription.class) != null;
        }
        
    }
    
    private static class ProjectNode extends AbstractNode {
        private final Node rootNode;
        
        ProjectNode(Children children, Node rootNode) {
            super(children);
            this.rootNode=rootNode;
            setName(rootNode.getDisplayName());
        }
        
        @Override
        public Image getIcon(int type) {
            return rootNode.getIcon(type);
        }
        
        @Override
        public Image getOpenedIcon(int type) {
            return rootNode.getOpenedIcon(type);
        }
    }

}
