/*
 * Decompiled with CFR 0.152.
 */
package org.apache.unomi.services.impl.cluster;

import com.sun.management.OperatingSystemMXBean;
import java.io.Serializable;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.unomi.api.ClusterNode;
import org.apache.unomi.api.Item;
import org.apache.unomi.api.PartialList;
import org.apache.unomi.api.ServerInfo;
import org.apache.unomi.api.conditions.Condition;
import org.apache.unomi.api.conditions.ConditionType;
import org.apache.unomi.api.services.ClusterService;
import org.apache.unomi.lifecycle.BundleWatcher;
import org.apache.unomi.persistence.spi.PersistenceService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterServiceImpl
implements ClusterService {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)ClusterServiceImpl.class.getName());
    private PersistenceService persistenceService;
    private String publicAddress;
    private String internalAddress;
    private ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(3);
    private String nodeId;
    private long nodeStartTime;
    private long nodeStatisticsUpdateFrequency = 10000L;
    private Map<String, Map<String, Serializable>> nodeSystemStatistics = new ConcurrentHashMap<String, Map<String, Serializable>>();
    private volatile boolean shutdownNow = false;
    private volatile List<ClusterNode> cachedClusterNodes = Collections.emptyList();
    private BundleWatcher bundleWatcher;
    private ScheduledFuture<?> updateSystemStatsFuture;
    private ScheduledFuture<?> cleanupStaleNodesFuture;
    private static final long MAX_WAIT_TIME = 60000L;

    public void setBundleWatcher(BundleWatcher bundleWatcher) {
        this.bundleWatcher = bundleWatcher;
        LOGGER.info("BundleWatcher service set");
    }

    private void waitForPersistenceService() {
        if (this.shutdownNow) {
            return;
        }
        if (this.persistenceService != null) {
            LOGGER.debug("Persistence service is already available, no need to wait");
            return;
        }
        long startTime = System.currentTimeMillis();
        long waitTime = 50L;
        while (System.currentTimeMillis() - startTime < 60000L) {
            if (this.persistenceService != null) {
                LOGGER.info("Persistence service is now available");
                return;
            }
            try {
                LOGGER.debug("Waiting for persistence service... ({}ms elapsed)", (Object)(System.currentTimeMillis() - startTime));
                Thread.sleep(waitTime);
                waitTime = Math.min(waitTime * 2L, 5000L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.error("Interrupted while waiting for persistence service", (Throwable)e);
                break;
            }
        }
        throw new IllegalStateException("PersistenceService not available after waiting 60000ms");
    }

    public void setPersistenceService(PersistenceService persistenceService) {
        this.persistenceService = persistenceService;
        LOGGER.info("PersistenceService set directly");
    }

    public void setPublicAddress(String publicAddress) {
        this.publicAddress = publicAddress;
    }

    public void setInternalAddress(String internalAddress) {
        this.internalAddress = internalAddress;
    }

    public void setNodeStatisticsUpdateFrequency(long nodeStatisticsUpdateFrequency) {
        this.nodeStatisticsUpdateFrequency = nodeStatisticsUpdateFrequency;
    }

    public void setNodeId(String nodeId) {
        this.nodeId = nodeId;
    }

    public Map<String, Map<String, Serializable>> getNodeSystemStatistics() {
        return this.nodeSystemStatistics;
    }

    public void init() {
        if (StringUtils.isBlank((CharSequence)this.nodeId)) {
            String errorMessage = "CRITICAL: nodeId is not set. This is a required setting for cluster operation.";
            LOGGER.error(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
        try {
            this.waitForPersistenceService();
        }
        catch (IllegalStateException e) {
            LOGGER.error("Failed to initialize cluster service: {}", (Object)e.getMessage());
            return;
        }
        this.nodeStartTime = System.currentTimeMillis();
        this.registerNodeInPersistence();
        this.initializeScheduledTasks();
        LOGGER.info("Cluster service initialized with node ID: {}", (Object)this.nodeId);
    }

    public void initializeScheduledTasks() {
        TimerTask statisticsTask = new TimerTask(){

            @Override
            public void run() {
                try {
                    ClusterServiceImpl.this.updateSystemStats();
                }
                catch (Throwable t) {
                    LOGGER.error("Error updating system statistics", t);
                }
            }
        };
        this.updateSystemStatsFuture = this.scheduledExecutorService.scheduleAtFixedRate(statisticsTask, 100L, this.nodeStatisticsUpdateFrequency, TimeUnit.MILLISECONDS);
        TimerTask cleanupTask = new TimerTask(){

            @Override
            public void run() {
                try {
                    ClusterServiceImpl.this.cleanupStaleNodes();
                }
                catch (Throwable t) {
                    LOGGER.error("Error cleaning up stale nodes", t);
                }
            }
        };
        this.cleanupStaleNodesFuture = this.scheduledExecutorService.scheduleAtFixedRate(cleanupTask, 100L, 60000L, TimeUnit.MILLISECONDS);
        LOGGER.info("Cluster service scheduled tasks initialized");
    }

    public void destroy() {
        boolean successfullyCancelled;
        LOGGER.info("Cluster service shutting down...");
        this.shutdownNow = true;
        if (this.updateSystemStatsFuture != null) {
            successfullyCancelled = this.updateSystemStatsFuture.cancel(false);
            if (!successfullyCancelled) {
                LOGGER.warn("Failed to cancel scheduled task: clusterNodeStatisticsUpdate");
            } else {
                LOGGER.info("Scheduled task: clusterNodeStatisticsUpdate cancelled");
            }
        }
        if (this.cleanupStaleNodesFuture != null) {
            successfullyCancelled = this.cleanupStaleNodesFuture.cancel(false);
            if (!successfullyCancelled) {
                LOGGER.warn("Failed to cancel scheduled task: cleanupStaleNodesFuture");
            } else {
                LOGGER.info("Scheduled task: cleanupStaleNodesFuture cancelled");
            }
        }
        if (this.scheduledExecutorService != null) {
            this.scheduledExecutorService.shutdownNow();
            try {
                boolean successfullyTerminated = this.scheduledExecutorService.awaitTermination(10L, TimeUnit.SECONDS);
                if (!successfullyTerminated) {
                    LOGGER.warn("Failed to terminate scheduled tasks after 10 seconds...");
                } else {
                    LOGGER.info("Scheduled tasks terminated");
                }
            }
            catch (InterruptedException e) {
                LOGGER.error("Error waiting for scheduled tasks to terminate", (Throwable)e);
            }
        }
        if (this.persistenceService != null) {
            try {
                this.persistenceService.remove(this.nodeId, ClusterNode.class);
                LOGGER.info("Node {} removed from cluster", (Object)this.nodeId);
            }
            catch (Exception e) {
                LOGGER.error("Error removing node from cluster", (Throwable)e);
            }
        }
        this.persistenceService = null;
        this.bundleWatcher = null;
        LOGGER.info("Cluster service shutdown.");
    }

    private void registerNodeInPersistence() {
        if (this.persistenceService == null) {
            LOGGER.error("Cannot register node: PersistenceService not available");
            return;
        }
        ClusterNode clusterNode = new ClusterNode();
        clusterNode.setItemId(this.nodeId);
        clusterNode.setPublicHostAddress(this.publicAddress);
        clusterNode.setInternalHostAddress(this.internalAddress);
        clusterNode.setStartTime(this.nodeStartTime);
        clusterNode.setLastHeartbeat(System.currentTimeMillis());
        if (this.bundleWatcher != null && !this.bundleWatcher.getServerInfos().isEmpty()) {
            ServerInfo serverInfo = (ServerInfo)this.bundleWatcher.getServerInfos().get(0);
            clusterNode.setServerInfo(serverInfo);
            LOGGER.info("Added server info to node: version={}, build={}", (Object)serverInfo.getServerVersion(), (Object)serverInfo.getServerBuildNumber());
        } else {
            LOGGER.warn("BundleWatcher not available at registration time, server info will not be available");
        }
        this.updateSystemStatsForNode(clusterNode);
        boolean success = this.persistenceService.save((Item)clusterNode);
        if (success) {
            LOGGER.info("Node {} registered in cluster", (Object)this.nodeId);
        } else {
            LOGGER.error("Failed to register node {} in cluster", (Object)this.nodeId);
        }
    }

    private void updateSystemStatsForNode(ClusterNode node) {
        RuntimeMXBean remoteRuntime = ManagementFactory.getRuntimeMXBean();
        long uptime = remoteRuntime.getUptime();
        double systemCpuLoad = 0.0;
        try {
            systemCpuLoad = ((OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean()).getSystemCpuLoad();
            if (Double.isNaN(systemCpuLoad)) {
                LOGGER.debug("System CPU load is NaN, setting to 0.0");
                systemCpuLoad = 0.0;
            }
        }
        catch (Exception e) {
            LOGGER.debug("Error retrieving system CPU load", (Throwable)e);
        }
        java.lang.management.OperatingSystemMXBean operatingSystemMXBean = ManagementFactory.getOperatingSystemMXBean();
        double systemLoadAverage = operatingSystemMXBean.getSystemLoadAverage();
        if (Double.isNaN(systemLoadAverage)) {
            LOGGER.debug("System load average is NaN, setting to 0.0");
            systemLoadAverage = 0.0;
        }
        node.setCpuLoad(systemCpuLoad);
        node.setUptime(uptime);
        ArrayList<Double> systemLoadAverageArray = new ArrayList<Double>();
        systemLoadAverageArray.add(systemLoadAverage);
        node.setLoadAverage(ArrayUtils.toPrimitive((Double[])systemLoadAverageArray.toArray(new Double[0])));
        TreeMap<String, Serializable> systemStatistics = new TreeMap<String, Serializable>();
        systemStatistics.put("systemLoadAverage", systemLoadAverageArray);
        systemStatistics.put("systemCpuLoad", Double.valueOf(systemCpuLoad));
        systemStatistics.put("uptime", Long.valueOf(uptime));
        this.nodeSystemStatistics.put(this.nodeId, systemStatistics);
    }

    private void updateSystemStats() {
        if (this.shutdownNow) {
            return;
        }
        if (this.persistenceService == null) {
            LOGGER.warn("Cannot update system stats: PersistenceService not available");
            return;
        }
        ClusterNode node = (ClusterNode)this.persistenceService.load(this.nodeId, ClusterNode.class);
        if (node == null) {
            LOGGER.warn("Node {} not found in persistence, re-registering", (Object)this.nodeId);
            this.registerNodeInPersistence();
            return;
        }
        try {
            this.updateSystemStatsForNode(node);
            if (this.bundleWatcher != null && !this.bundleWatcher.getServerInfos().isEmpty()) {
                ServerInfo currentInfo = (ServerInfo)this.bundleWatcher.getServerInfos().get(0);
                if (node.getServerInfo() == null || !currentInfo.getServerVersion().equals(node.getServerInfo().getServerVersion())) {
                    node.setServerInfo(currentInfo);
                    LOGGER.info("Updated server info for node {}: version={}, build={}", new Object[]{this.nodeId, currentInfo.getServerVersion(), currentInfo.getServerBuildNumber()});
                }
            }
            node.setLastHeartbeat(System.currentTimeMillis());
            boolean success = this.persistenceService.save((Item)node);
            if (!success) {
                LOGGER.error("Failed to update node {} statistics", (Object)this.nodeId);
            }
            try {
                List nodes;
                this.cachedClusterNodes = nodes = this.persistenceService.getAllItems(ClusterNode.class, 0, -1, null).getList();
            }
            catch (Exception e) {
                LOGGER.warn("Failed to refresh cluster nodes cache during stats update", (Throwable)e);
            }
        }
        catch (Exception e) {
            LOGGER.error("Error updating system statistics for node {}: {}", new Object[]{this.nodeId, e.getMessage(), e});
        }
    }

    private void cleanupStaleNodes() {
        if (this.shutdownNow) {
            return;
        }
        if (this.persistenceService == null) {
            LOGGER.warn("Cannot cleanup stale nodes: PersistenceService not available");
            return;
        }
        long cutoffTime = System.currentTimeMillis() - this.nodeStatisticsUpdateFrequency * 3L;
        Condition staleNodesCondition = new Condition();
        ConditionType propertyConditionType = new ConditionType();
        propertyConditionType.setItemId("propertyCondition");
        propertyConditionType.setItemType("conditionType");
        propertyConditionType.setConditionEvaluator("propertyConditionEvaluator");
        propertyConditionType.setQueryBuilder("propertyConditionESQueryBuilder");
        staleNodesCondition.setConditionType(propertyConditionType);
        staleNodesCondition.setConditionTypeId("propertyCondition");
        staleNodesCondition.setParameter("propertyName", (Object)"lastHeartbeat");
        staleNodesCondition.setParameter("comparisonOperator", (Object)"lessThan");
        staleNodesCondition.setParameter("propertyValueInteger", (Object)cutoffTime);
        PartialList staleNodes = this.persistenceService.query(staleNodesCondition, null, ClusterNode.class, 0, -1);
        for (ClusterNode staleNode : staleNodes.getList()) {
            LOGGER.info("Removing stale node: {}", (Object)staleNode.getItemId());
            this.persistenceService.remove(staleNode.getItemId(), ClusterNode.class);
            this.nodeSystemStatistics.remove(staleNode.getItemId());
        }
    }

    public List<ClusterNode> getClusterNodes() {
        return this.cachedClusterNodes.isEmpty() ? Collections.emptyList() : new ArrayList<ClusterNode>(this.cachedClusterNodes);
    }

    public void purge(Date date) {
        if (this.persistenceService == null) {
            LOGGER.warn("Cannot purge by date: PersistenceService not available");
            return;
        }
        this.persistenceService.purge(date);
    }

    public void purge(String scope) {
        if (this.persistenceService == null) {
            LOGGER.warn("Cannot purge by scope: PersistenceService not available");
            return;
        }
        this.persistenceService.purge(scope);
    }

    public boolean isPersistenceServiceAvailable() {
        return this.persistenceService != null;
    }
}

