/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.metrics;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.db.rows.RangeTombstoneMarker;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.db.transform.Transformation;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.Clock;

public class TopPartitionTracker
implements Closeable {
    private static final String SIZES = "SIZES";
    private static final String TOMBSTONES = "TOMBSTONES";
    private final AtomicReference<TopHolder> topSizes = new AtomicReference();
    private final AtomicReference<TopHolder> topTombstones = new AtomicReference();
    private final TableMetadata metadata;
    private final Future<?> scheduledSave;
    private long lastTombstoneSave = 0L;
    private long lastSizeSave = 0L;
    private static final Comparator<TopPartition> comparator = (o1, o2) -> {
        int cmp = -Long.compare(o1.value, o2.value);
        if (cmp != 0) {
            return cmp;
        }
        return o1.key.compareTo(o2.key);
    };

    public TopPartitionTracker(TableMetadata metadata) {
        this.metadata = metadata;
        this.topSizes.set(new TopHolder(SystemKeyspace.getTopPartitions(metadata, SIZES), DatabaseDescriptor.getMaxTopSizePartitionCount(), DatabaseDescriptor.getMinTrackedPartitionSizeInBytes().toBytes()));
        this.topTombstones.set(new TopHolder(SystemKeyspace.getTopPartitions(metadata, TOMBSTONES), DatabaseDescriptor.getMaxTopTombstonePartitionCount(), DatabaseDescriptor.getMinTrackedPartitionTombstoneCount()));
        this.scheduledSave = ScheduledExecutors.optionalTasks.scheduleAtFixedRate(this::save, 60L, 60L, TimeUnit.MINUTES);
    }

    @Override
    public void close() {
        this.scheduledSave.cancel(true);
    }

    @VisibleForTesting
    public void save() {
        TopHolder sizes = this.topSizes.get();
        if (!sizes.top.isEmpty() && sizes.lastUpdate > this.lastSizeSave) {
            SystemKeyspace.saveTopPartitions(this.metadata, SIZES, sizes.top, sizes.lastUpdate);
            this.lastSizeSave = sizes.lastUpdate;
        }
        TopHolder tombstones = this.topTombstones.get();
        if (!tombstones.top.isEmpty() && tombstones.lastUpdate > this.lastTombstoneSave) {
            SystemKeyspace.saveTopPartitions(this.metadata, TOMBSTONES, tombstones.top, tombstones.lastUpdate);
            this.lastTombstoneSave = tombstones.lastUpdate;
        }
    }

    public void merge(Collector collector) {
        TopHolder newTombstones;
        TopHolder newSizes;
        TopHolder cur;
        while (!this.topSizes.compareAndSet(cur = this.topSizes.get(), newSizes = cur.merge(collector.sizes, StorageService.instance.getLocalReplicas(this.metadata.keyspace).ranges()))) {
        }
        while (!this.topTombstones.compareAndSet(cur = this.topTombstones.get(), newTombstones = cur.merge(collector.tombstones, StorageService.instance.getLocalReplicas(this.metadata.keyspace).ranges()))) {
        }
    }

    public String toString() {
        return "TopPartitionTracker:\ntopSizes:\n" + this.topSizes.get() + '\n' + "topTombstones:\n" + this.topTombstones.get() + '\n';
    }

    public Map<String, Long> getTopTombstonePartitionMap() {
        return this.topTombstones.get().toMap(this.metadata);
    }

    public Map<String, Long> getTopSizePartitionMap() {
        return this.topSizes.get().toMap(this.metadata);
    }

    @VisibleForTesting
    public TopHolder topSizes() {
        return this.topSizes.get();
    }

    @VisibleForTesting
    public TopHolder topTombstones() {
        return this.topTombstones.get();
    }

    public static class StoredTopPartitions {
        public static StoredTopPartitions EMPTY = new StoredTopPartitions(Collections.emptyList(), 0L);
        public final List<TopPartition> topPartitions;
        public final long lastUpdated;

        public StoredTopPartitions(List<TopPartition> topPartitions, long lastUpdated) {
            this.topPartitions = topPartitions;
            this.lastUpdated = lastUpdated;
        }
    }

    public static class TombstoneCounter
    extends Transformation<UnfilteredRowIterator> {
        private final Collector collector;
        private final int nowInSec;
        private long tombstoneCount = 0L;
        private DecoratedKey key = null;

        public TombstoneCounter(Collector collector, int nowInSec) {
            this.collector = collector;
            this.nowInSec = nowInSec;
        }

        @Override
        public Row applyToRow(Row row) {
            if (!row.deletion().isLive()) {
                ++this.tombstoneCount;
            }
            if (row.hasDeletion(this.nowInSec)) {
                for (Cell<?> c : row.cells()) {
                    if (!c.isTombstone()) continue;
                    ++this.tombstoneCount;
                }
            }
            return row;
        }

        @Override
        public RangeTombstoneMarker applyToMarker(RangeTombstoneMarker marker) {
            ++this.tombstoneCount;
            return marker;
        }

        @Override
        protected UnfilteredRowIterator applyToPartition(UnfilteredRowIterator partition) {
            this.reset(partition.partitionKey());
            if (!partition.partitionLevelDeletion().isLive()) {
                ++this.tombstoneCount;
            }
            return Transformation.apply(partition, this);
        }

        private void reset(DecoratedKey key) {
            this.tombstoneCount = 0L;
            this.key = key;
        }

        @Override
        public void onPartitionClose() {
            this.collector.trackTombstoneCount(this.key, this.tombstoneCount);
        }
    }

    public static class TopPartition
    implements Comparable<TopPartition> {
        public final DecoratedKey key;
        public final long value;

        public TopPartition(DecoratedKey key, long value) {
            this.key = key;
            this.value = value;
        }

        @Override
        public int compareTo(TopPartition o) {
            return comparator.compare(this, o);
        }

        public String toString() {
            return "TopPartition{key=" + this.key + ", value=" + this.value + '}';
        }
    }

    public static class TopHolder {
        public final NavigableSet<TopPartition> top;
        private final int maxTopPartitionCount;
        private final long minTrackedValue;
        private final Collection<Range<Token>> ranges;
        private long currentMinValue = Long.MAX_VALUE;
        public final long lastUpdate;

        private TopHolder(int maxTopPartitionCount, long minTrackedValue, Collection<Range<Token>> ranges) {
            this(maxTopPartitionCount, minTrackedValue, new TreeSet<TopPartition>(), ranges, 0L);
        }

        private TopHolder(int maxTopPartitionCount, long minTrackedValue, NavigableSet<TopPartition> top, Collection<Range<Token>> ranges, long lastUpdate) {
            this.maxTopPartitionCount = maxTopPartitionCount;
            this.minTrackedValue = minTrackedValue;
            this.top = top;
            this.ranges = ranges;
            this.lastUpdate = lastUpdate;
        }

        private TopHolder(StoredTopPartitions storedTopPartitions, int maxTopPartitionCount, long minTrackedValue) {
            this.maxTopPartitionCount = maxTopPartitionCount;
            this.minTrackedValue = minTrackedValue;
            this.top = new TreeSet<TopPartition>();
            this.ranges = null;
            this.lastUpdate = storedTopPartitions.lastUpdated;
            for (TopPartition topPartition : storedTopPartitions.topPartitions) {
                this.track(topPartition);
            }
        }

        public void track(DecoratedKey key, long value) {
            if (value < this.minTrackedValue) {
                return;
            }
            if (this.top.size() < this.maxTopPartitionCount || value > this.currentMinValue) {
                this.track(new TopPartition(SSTable.getMinimalKey(key), value));
            }
        }

        private void track(TopPartition tp) {
            this.top.add(tp);
            while (this.top.size() > this.maxTopPartitionCount) {
                this.top.pollLast();
                this.currentMinValue = ((TopPartition)this.top.last()).value;
            }
            this.currentMinValue = Math.min(tp.value, this.currentMinValue);
        }

        public TopHolder merge(TopHolder holder, Collection<Range<Token>> ownedRanges) {
            TopHolder mergedHolder = holder.cloneForMerging(Clock.Global.currentTimeMillis());
            for (TopPartition existingTop : this.top) {
                if (Range.isInRanges(existingTop.key.getToken(), mergedHolder.ranges) || !ownedRanges.isEmpty() && !Range.isInRanges(existingTop.key.getToken(), ownedRanges)) continue;
                mergedHolder.track(existingTop);
            }
            return mergedHolder;
        }

        private TopHolder cloneForMerging(long lastUpdate) {
            return new TopHolder(this.maxTopPartitionCount, this.minTrackedValue, new TreeSet<TopPartition>((SortedSet<TopPartition>)this.top), this.ranges, lastUpdate);
        }

        public String toString() {
            int i = 0;
            Iterator<TopPartition> it = this.top.iterator();
            StringBuilder sb = new StringBuilder();
            while (it.hasNext()) {
                sb.append(++i).append(':').append(it.next()).append(System.lineSeparator());
            }
            return sb.toString();
        }

        public Map<String, Long> toMap(TableMetadata metadata) {
            LinkedHashMap<String, Long> topPartitionsMap = new LinkedHashMap<String, Long>();
            for (TopPartition topPartition : this.top) {
                String key = metadata.partitionKeyType.getString(topPartition.key.getKey());
                topPartitionsMap.put(key, topPartition.value);
            }
            return topPartitionsMap;
        }
    }

    public static class Collector {
        private final TopHolder tombstones;
        private final TopHolder sizes;

        public Collector(Collection<Range<Token>> ranges) {
            this.tombstones = new TopHolder(DatabaseDescriptor.getMaxTopTombstonePartitionCount(), DatabaseDescriptor.getMinTrackedPartitionTombstoneCount(), ranges);
            this.sizes = new TopHolder(DatabaseDescriptor.getMaxTopSizePartitionCount(), DatabaseDescriptor.getMinTrackedPartitionSizeInBytes().toBytes(), ranges);
        }

        public void trackTombstoneCount(DecoratedKey key, long count) {
            this.tombstones.track(key, count);
        }

        public void trackPartitionSize(DecoratedKey key, long size) {
            this.sizes.track(key, size);
        }

        public String toString() {
            return "tombstones:\n" + this.tombstones + "\nsizes:\n" + this.sizes;
        }
    }
}

