/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.common.config;

import com.google.common.collect.Maps;
import io.netty.buffer.PooledByteBufAllocator;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.rocketmq.common.ThreadFactoryImpl;
import org.apache.rocketmq.common.utils.ThreadUtils;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.CompactRangeOptions;
import org.rocksdb.CompactionOptions;
import org.rocksdb.CompressionType;
import org.rocksdb.DBOptions;
import org.rocksdb.Env;
import org.rocksdb.FlushOptions;
import org.rocksdb.LiveFileMetaData;
import org.rocksdb.Priority;
import org.rocksdb.ReadOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.Statistics;
import org.rocksdb.Status;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;

public abstract class AbstractRocksDBStorage {
    protected static final Logger LOGGER = LoggerFactory.getLogger((String)"RocketmqRocksDB");
    public static final PooledByteBufAllocator POOLED_ALLOCATOR = new PooledByteBufAllocator(true);
    public static final byte CTRL_0 = 0;
    public static final byte CTRL_1 = 1;
    public static final byte CTRL_2 = 2;
    private static final String SPACE = " | ";
    protected final String dbPath;
    protected boolean readOnly;
    protected RocksDB db;
    protected DBOptions options;
    protected WriteOptions writeOptions;
    protected WriteOptions ableWalWriteOptions;
    protected ReadOptions readOptions;
    protected ReadOptions totalOrderReadOptions;
    protected CompactionOptions compactionOptions;
    protected CompactRangeOptions compactRangeOptions;
    protected ColumnFamilyHandle defaultCFHandle;
    protected final List<ColumnFamilyOptions> cfOptions = new ArrayList<ColumnFamilyOptions>();
    protected final List<ColumnFamilyHandle> cfHandles = new ArrayList<ColumnFamilyHandle>();
    protected volatile boolean loaded;
    protected CompressionType compressionType = CompressionType.LZ4_COMPRESSION;
    private volatile boolean closed;
    private final Semaphore reloadPermit = new Semaphore(1);
    private final ScheduledExecutorService reloadScheduler = ThreadUtils.newScheduledThreadPool(1, new ThreadFactoryImpl("RocksDBStorageReloadService_"));
    private final ThreadPoolExecutor manualCompactionThread = (ThreadPoolExecutor)ThreadUtils.newThreadPoolExecutor(1, 1, 60000L, TimeUnit.MILLISECONDS, new ArrayBlockingQueue<Runnable>(1), new ThreadFactoryImpl("RocksDBManualCompactionService_"), new ThreadPoolExecutor.DiscardOldestPolicy());

    public AbstractRocksDBStorage(String dbPath) {
        this.dbPath = dbPath;
    }

    protected void initOptions() {
        this.initWriteOptions();
        this.initAbleWalWriteOptions();
        this.initReadOptions();
        this.initTotalOrderReadOptions();
        this.initCompactRangeOptions();
        this.initCompactionOptions();
    }

    protected void initWriteOptions() {
        this.writeOptions = new WriteOptions();
        this.writeOptions.setSync(false);
        this.writeOptions.setDisableWAL(true);
        this.writeOptions.setNoSlowdown(false);
    }

    protected void initAbleWalWriteOptions() {
        this.ableWalWriteOptions = new WriteOptions();
        this.ableWalWriteOptions.setSync(false);
        this.ableWalWriteOptions.setDisableWAL(false);
        this.ableWalWriteOptions.setNoSlowdown(false);
    }

    protected void initReadOptions() {
        this.readOptions = new ReadOptions();
        this.readOptions.setPrefixSameAsStart(true);
        this.readOptions.setTotalOrderSeek(false);
        this.readOptions.setTailing(false);
    }

    protected void initTotalOrderReadOptions() {
        this.totalOrderReadOptions = new ReadOptions();
        this.totalOrderReadOptions.setPrefixSameAsStart(false);
        this.totalOrderReadOptions.setTotalOrderSeek(true);
        this.totalOrderReadOptions.setTailing(false);
    }

    protected void initCompactRangeOptions() {
        this.compactRangeOptions = new CompactRangeOptions();
        this.compactRangeOptions.setBottommostLevelCompaction(CompactRangeOptions.BottommostLevelCompaction.kForce);
        this.compactRangeOptions.setAllowWriteStall(true);
        this.compactRangeOptions.setExclusiveManualCompaction(false);
        this.compactRangeOptions.setChangeLevel(true);
        this.compactRangeOptions.setTargetLevel(-1);
        this.compactRangeOptions.setMaxSubcompactions(4);
    }

    protected void initCompactionOptions() {
        this.compactionOptions = new CompactionOptions();
        this.compactionOptions.setCompression(this.compressionType);
        this.compactionOptions.setMaxSubcompactions(4);
        this.compactionOptions.setOutputFileSizeLimit(0x100000000L);
    }

    public boolean hold() {
        if (!this.loaded || this.db == null || this.closed) {
            LOGGER.error("hold rocksdb Failed. {}", (Object)this.dbPath);
            return false;
        }
        return true;
    }

    public void release() {
    }

    protected void put(ColumnFamilyHandle cfHandle, WriteOptions writeOptions, byte[] keyBytes, int keyLen, byte[] valueBytes, int valueLen) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            this.db.put(cfHandle, writeOptions, keyBytes, 0, keyLen, valueBytes, 0, valueLen);
        }
        catch (RocksDBException e) {
            this.scheduleReloadRocksdb(e);
            LOGGER.error("put Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    protected void put(ColumnFamilyHandle cfHandle, WriteOptions writeOptions, ByteBuffer keyBB, ByteBuffer valueBB) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            this.db.put(cfHandle, writeOptions, keyBB, valueBB);
        }
        catch (RocksDBException e) {
            this.scheduleReloadRocksdb(e);
            LOGGER.error("put Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    protected void batchPut(WriteOptions writeOptions, WriteBatch batch) throws RocksDBException {
        try {
            this.db.write(writeOptions, batch);
        }
        catch (RocksDBException e) {
            this.scheduleReloadRocksdb(e);
            LOGGER.error("batchPut Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            batch.clear();
        }
    }

    protected byte[] get(ColumnFamilyHandle cfHandle, ReadOptions readOptions, byte[] keyBytes) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            byte[] byArray = this.db.get(cfHandle, readOptions, keyBytes);
            return byArray;
        }
        catch (RocksDBException e) {
            LOGGER.error("get Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    protected int get(ColumnFamilyHandle cfHandle, ReadOptions readOptions, ByteBuffer keyBB, ByteBuffer valueBB) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            int n = this.db.get(cfHandle, readOptions, keyBB, valueBB);
            return n;
        }
        catch (RocksDBException e) {
            LOGGER.error("get Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    protected List<byte[]> multiGet(ReadOptions readOptions, List<ColumnFamilyHandle> columnFamilyHandleList, List<byte[]> keys) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            List list = this.db.multiGetAsList(readOptions, columnFamilyHandleList, keys);
            return list;
        }
        catch (RocksDBException e) {
            LOGGER.error("multiGet Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    protected void delete(ColumnFamilyHandle cfHandle, WriteOptions writeOptions, byte[] keyBytes) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            this.db.delete(cfHandle, writeOptions, keyBytes);
        }
        catch (RocksDBException e) {
            LOGGER.error("delete Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    protected void delete(ColumnFamilyHandle cfHandle, WriteOptions writeOptions, ByteBuffer keyBB) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            this.db.delete(cfHandle, writeOptions, keyBB);
        }
        catch (RocksDBException e) {
            LOGGER.error("delete Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    protected void rangeDelete(ColumnFamilyHandle cfHandle, WriteOptions writeOptions, byte[] startKey, byte[] endKey) throws RocksDBException {
        if (!this.hold()) {
            throw new IllegalStateException("rocksDB:" + this + " is not ready");
        }
        try {
            this.db.deleteRange(cfHandle, writeOptions, startKey, endKey);
        }
        catch (RocksDBException e) {
            this.scheduleReloadRocksdb(e);
            LOGGER.error("rangeDelete Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
        finally {
            this.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void manualCompactionDefaultCfRange(CompactRangeOptions compactRangeOptions) {
        if (!this.hold()) {
            return;
        }
        long s1 = System.currentTimeMillis();
        boolean result = true;
        try {
            LOGGER.info("manualCompaction Start. {}", (Object)this.dbPath);
            this.db.compactRange(this.defaultCFHandle, null, null, compactRangeOptions);
            this.release();
        }
        catch (RocksDBException e) {
            try {
                result = false;
                this.scheduleReloadRocksdb(e);
                LOGGER.error("manualCompaction Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
                this.release();
            }
            catch (Throwable throwable) {
                this.release();
                LOGGER.info("manualCompaction End. {}, rt: {}(ms), result: {}", new Object[]{this.dbPath, System.currentTimeMillis() - s1, result});
                throw throwable;
            }
            LOGGER.info("manualCompaction End. {}, rt: {}(ms), result: {}", new Object[]{this.dbPath, System.currentTimeMillis() - s1, result});
        }
        LOGGER.info("manualCompaction End. {}, rt: {}(ms), result: {}", new Object[]{this.dbPath, System.currentTimeMillis() - s1, result});
    }

    protected void manualCompaction(long minPhyOffset, final CompactRangeOptions compactRangeOptions) {
        this.manualCompactionThread.submit(new Runnable(){

            @Override
            public void run() {
                AbstractRocksDBStorage.this.manualCompactionDefaultCfRange(compactRangeOptions);
            }
        });
    }

    protected void open(List<ColumnFamilyDescriptor> cfDescriptors) throws RocksDBException {
        this.cfHandles.clear();
        this.db = this.readOnly ? RocksDB.openReadOnly((DBOptions)this.options, (String)this.dbPath, cfDescriptors, this.cfHandles) : RocksDB.open((DBOptions)this.options, (String)this.dbPath, cfDescriptors, this.cfHandles);
        assert (cfDescriptors.size() == this.cfHandles.size());
        if (this.db == null) {
            throw new RocksDBException("open rocksdb null");
        }
        try (Env env = this.db.getEnv();){
            env.setBackgroundThreads(8, Priority.LOW);
        }
    }

    protected abstract boolean postLoad();

    public synchronized boolean start() {
        if (this.loaded) {
            return true;
        }
        if (this.postLoad()) {
            this.loaded = true;
            LOGGER.info("RocksDB [{}] starts OK", (Object)this.dbPath);
            this.closed = false;
            return true;
        }
        return false;
    }

    protected abstract void preShutdown();

    public synchronized boolean shutdown() {
        try {
            if (!this.loaded) {
                LOGGER.info("RocksDBStorage is not loaded, shutdown OK. dbPath={}, readOnly={}", (Object)this.dbPath, (Object)this.readOnly);
                return true;
            }
            FlushOptions flushOptions = new FlushOptions();
            flushOptions.setWaitForFlush(true);
            try {
                this.flush(flushOptions);
            }
            finally {
                flushOptions.close();
            }
            this.db.cancelAllBackgroundWork(true);
            this.db.pauseBackgroundWork();
            this.preShutdown();
            this.defaultCFHandle.close();
            for (ColumnFamilyOptions opt : this.cfOptions) {
                opt.close();
            }
            if (this.writeOptions != null) {
                this.writeOptions.close();
            }
            if (this.ableWalWriteOptions != null) {
                this.ableWalWriteOptions.close();
            }
            if (this.readOptions != null) {
                this.readOptions.close();
            }
            if (this.totalOrderReadOptions != null) {
                this.totalOrderReadOptions.close();
            }
            if (this.db != null && !this.readOnly) {
                this.db.syncWal();
            }
            if (this.db != null) {
                this.db.closeE();
            }
            if (this.options != null) {
                this.options.close();
            }
            this.cfOptions.clear();
            this.db = null;
            this.readOptions = null;
            this.totalOrderReadOptions = null;
            this.writeOptions = null;
            this.ableWalWriteOptions = null;
            this.options = null;
            this.loaded = false;
            LOGGER.info("RocksDB shutdown OK. {}", (Object)this.dbPath);
        }
        catch (Exception e) {
            LOGGER.error("RocksDB shutdown failed. {}", (Object)this.dbPath, (Object)e);
            return false;
        }
        return true;
    }

    public void flush(FlushOptions flushOptions) throws RocksDBException {
        this.flush(flushOptions, this.cfHandles);
    }

    public void flush(FlushOptions flushOptions, List<ColumnFamilyHandle> columnFamilyHandles) throws RocksDBException {
        if (!this.loaded || this.readOnly || this.closed) {
            return;
        }
        try {
            if (this.db != null) {
                this.db.flush(flushOptions, columnFamilyHandles);
            }
        }
        catch (RocksDBException e) {
            this.scheduleReloadRocksdb(e);
            LOGGER.error("flush Failed. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(e));
            throw e;
        }
    }

    public void flushWAL() throws RocksDBException {
        this.db.flushWal(true);
    }

    public Statistics getStatistics() {
        return this.options.statistics();
    }

    public ColumnFamilyHandle getDefaultCFHandle() {
        return this.defaultCFHandle;
    }

    public List<LiveFileMetaData> getCompactionStatus() {
        if (!this.hold()) {
            return null;
        }
        try {
            List list = this.db.getLiveFilesMetaData();
            return list;
        }
        finally {
            this.release();
        }
    }

    private void scheduleReloadRocksdb(RocksDBException rocksDBException) {
        if (rocksDBException == null || rocksDBException.getStatus() == null) {
            return;
        }
        Status status = rocksDBException.getStatus();
        Status.Code code = status.getCode();
        if (Status.Code.Aborted == code || Status.Code.Corruption == code || Status.Code.Undefined == code) {
            LOGGER.error("scheduleReloadRocksdb. {}, {}", (Object)this.dbPath, (Object)this.getStatusError(rocksDBException));
            this.scheduleReloadRocksdb0();
        }
    }

    private void scheduleReloadRocksdb0() {
        if (!this.reloadPermit.tryAcquire()) {
            return;
        }
        this.closed = true;
        this.reloadScheduler.schedule(new Runnable(){

            @Override
            public void run() {
                boolean result = true;
                try {
                    AbstractRocksDBStorage.this.reloadRocksdb();
                }
                catch (Exception e) {
                    result = false;
                }
                finally {
                    AbstractRocksDBStorage.this.reloadPermit.release();
                }
                if (!result) {
                    LOGGER.info("reload rocksdb Retry. {}", (Object)AbstractRocksDBStorage.this.dbPath);
                    AbstractRocksDBStorage.this.scheduleReloadRocksdb0();
                }
            }
        }, 10L, TimeUnit.SECONDS);
    }

    private void reloadRocksdb() throws Exception {
        LOGGER.info("reload rocksdb Start. {}", (Object)this.dbPath);
        if (!this.shutdown() || !this.start()) {
            LOGGER.error("reload rocksdb Failed. {}", (Object)this.dbPath);
            throw new Exception("reload rocksdb Error");
        }
        LOGGER.info("reload rocksdb OK. {}", (Object)this.dbPath);
    }

    private String getStatusError(RocksDBException e) {
        if (e == null || e.getStatus() == null) {
            return "null";
        }
        Status status = e.getStatus();
        StringBuilder sb = new StringBuilder(64);
        sb.append("code: ");
        if (status.getCode() != null) {
            sb.append(status.getCode().name());
        } else {
            sb.append("null");
        }
        sb.append(", ").append("subCode: ");
        if (status.getSubCode() != null) {
            sb.append(status.getSubCode().name());
        } else {
            sb.append("null");
        }
        sb.append(", ").append("state: ").append(status.getState());
        return sb.toString();
    }

    public void statRocksdb(Logger logger) {
        try {
            String blockCacheMemUsage = this.db.getProperty("rocksdb.block-cache-usage");
            String indexesAndFilterBlockMemUsage = this.db.getProperty("rocksdb.estimate-table-readers-mem");
            String memTableMemUsage = this.db.getProperty("rocksdb.cur-size-all-mem-tables");
            String blocksPinnedByIteratorMemUsage = this.db.getProperty("rocksdb.block-cache-pinned-usage");
            logger.info("RocksDB Memory Usage: BlockCache: {}, IndexesAndFilterBlock: {}, MemTable: {}, BlocksPinnedByIterator: {}", new Object[]{blockCacheMemUsage, indexesAndFilterBlockMemUsage, memTableMemUsage, blocksPinnedByIteratorMemUsage});
            List<LiveFileMetaData> liveFileMetaDataList = this.getCompactionStatus();
            if (liveFileMetaDataList == null || liveFileMetaDataList.isEmpty()) {
                return;
            }
            HashMap map = Maps.newHashMap();
            for (LiveFileMetaData metaData : liveFileMetaDataList) {
                StringBuilder sb = map.computeIfAbsent(metaData.level(), k -> new StringBuilder(256));
                sb.append(new String(metaData.columnFamilyName(), StandardCharsets.UTF_8)).append(SPACE).append(metaData.fileName()).append(SPACE).append("file-size: ").append(metaData.size()).append(SPACE).append("number-of-entries: ").append(metaData.numEntries()).append(SPACE).append("file-read-times: ").append(metaData.numReadsSampled()).append(SPACE).append("deletions: ").append(metaData.numDeletions()).append(SPACE).append("being-compacted: ").append(metaData.beingCompacted()).append("\n");
            }
            map.forEach((key, value) -> logger.info("level: {}\n{}", key, (Object)value.toString()));
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    static {
        RocksDB.loadLibrary();
    }
}

