/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.agent.core.plugin.classloader;

import net.bytebuddy.ByteBuddy;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;

import static org.hamcrest.Matchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

class AgentPluginClassLoaderTest {
    
    @TempDir
    private Path tempDir;
    
    @Test
    void assertLoadClassWithManifestAndResources() throws IOException, ClassNotFoundException {
        Map<String, byte[]> pluginEntries = new HashMap<>(3, 1F);
        pluginEntries.put("org/apache/shardingsphere/agent/core/plugin/fixture/PluginFixture.class",
                createClassBytes("org.apache.shardingsphere.agent.core.plugin.fixture.PluginFixture"));
        pluginEntries.put("org/apache/shardingsphere/agent/core/plugin/fixture/AnotherPluginFixture.class",
                createClassBytes("org.apache.shardingsphere.agent.core.plugin.fixture.AnotherPluginFixture"));
        pluginEntries.put("fixture-resource.txt", "foo_resource".getBytes(StandardCharsets.UTF_8));
        Manifest manifest = createManifest();
        try (
                JarFile emptyJar = createJar("foo-empty.jar", new HashMap<>(1, 1F), createBasicManifest());
                JarFile pluginJar = createJar("foo-plugin.jar", pluginEntries, manifest)) {
            AgentPluginClassLoader classLoader = new AgentPluginClassLoader(new URLClassLoader(new URL[0], null), Arrays.asList(pluginJar, emptyJar));
            Class<?> loadedClass = classLoader.loadClass("org.apache.shardingsphere.agent.core.plugin.fixture.PluginFixture");
            Package loadedPackage = loadedClass.getPackage();
            assertThat(loadedPackage.getSpecificationTitle(), is("spec_title"));
            assertThat(loadedPackage.getSpecificationVersion(), is("1.0"));
            assertThat(loadedPackage.getSpecificationVendor(), is("spec_vendor"));
            assertThat(loadedPackage.getImplementationTitle(), is("impl_title"));
            assertThat(loadedPackage.getImplementationVersion(), is("2.0"));
            assertThat(loadedPackage.getImplementationVendor(), is("impl_vendor"));
            Class<?> anotherClass = classLoader.loadClass("org.apache.shardingsphere.agent.core.plugin.fixture.AnotherPluginFixture");
            assertNotNull(anotherClass);
            Enumeration<URL> resources = classLoader.findResources("fixture-resource.txt");
            assertTrue(resources.hasMoreElements());
            URL resourceUrl = resources.nextElement();
            assertThat(resourceUrl.toString(), is(String.format("jar:file:%s!/fixture-resource.txt", pluginJar.getName())));
            assertFalse(resources.hasMoreElements());
            URL directResource = classLoader.findResource("fixture-resource.txt");
            assertThat(directResource.toString(), is(String.format("jar:file:%s!/fixture-resource.txt", pluginJar.getName())));
        }
    }
    
    @Test
    void assertLoadClassWithoutPackage() throws IOException, ClassNotFoundException {
        Map<String, byte[]> entries = Collections.singletonMap("DefaultFixture.class", createClassBytes("DefaultFixture"));
        try (JarFile defaultJar = createJar("foo-default.jar", entries, createBasicManifest())) {
            AgentPluginClassLoader classLoader = new AgentPluginClassLoader(new URLClassLoader(new URL[0], null), Collections.singletonList(defaultJar));
            assertThat(classLoader.loadClass("DefaultFixture").getSimpleName(), is("DefaultFixture"));
        }
    }
    
    @Test
    void assertClassNotFoundWhenClassMissing() {
        AgentPluginClassLoader classLoader = new AgentPluginClassLoader(new URLClassLoader(new URL[0], null), new LinkedList<>());
        assertThrows(ClassNotFoundException.class, () -> classLoader.loadClass("org.apache.shardingsphere.agent.core.plugin.fixture.MissingClass"));
    }
    
    @Test
    void assertClassNotFoundWhenInputStreamBroken() throws IOException {
        try (JarFile ignored = createJar("foo-broken.jar", new HashMap<>(1, 1F), createBasicManifest())) {
            JarFile brokenJar = mock(JarFile.class);
            when(brokenJar.getEntry(anyString())).thenReturn(new ZipEntry("BrokenClass.class"));
            when(brokenJar.getManifest()).thenReturn(createBasicManifest());
            when(brokenJar.getInputStream(any(ZipEntry.class))).thenThrow(new IOException("broken"));
            AgentPluginClassLoader classLoader = new AgentPluginClassLoader(new URLClassLoader(new URL[0], null), Collections.singletonList(brokenJar));
            assertThrows(ClassNotFoundException.class, () -> classLoader.loadClass("BrokenClass"));
        }
    }
    
    @Test
    void assertFindResourceReturnsNullWhenUrlMalformed() throws IOException {
        Map<String, byte[]> entries = new HashMap<>(1, 1F);
        entries.put("malformed.txt", "bar_resource".getBytes(StandardCharsets.UTF_8));
        try (JarFile malformedJar = createJar("foo#malformed.jar", entries, createBasicManifest())) {
            Collection<JarFile> jars = new LinkedList<>();
            jars.add(malformedJar);
            AgentPluginClassLoader classLoader = new AgentPluginClassLoader(new URLClassLoader(new URL[0], null), jars);
            URL resourceUrl = classLoader.findResource("malformed.txt");
            assertNull(resourceUrl);
            assertFalse(classLoader.findResources("malformed.txt").hasMoreElements());
        }
    }
    
    private Manifest createManifest() {
        Manifest result = new Manifest();
        Attributes attributes = result.getMainAttributes();
        attributes.put(Attributes.Name.MANIFEST_VERSION, "1.0");
        attributes.put(Attributes.Name.SPECIFICATION_TITLE, "spec_title");
        attributes.put(Attributes.Name.SPECIFICATION_VERSION, "1.0");
        attributes.put(Attributes.Name.SPECIFICATION_VENDOR, "spec_vendor");
        attributes.put(Attributes.Name.IMPLEMENTATION_TITLE, "impl_title");
        attributes.put(Attributes.Name.IMPLEMENTATION_VERSION, "2.0");
        attributes.put(Attributes.Name.IMPLEMENTATION_VENDOR, "impl_vendor");
        return result;
    }
    
    private Manifest createBasicManifest() {
        Manifest result = new Manifest();
        result.getMainAttributes().put(Attributes.Name.MANIFEST_VERSION, "1.0");
        return result;
    }
    
    private JarFile createJar(final String fileName, final Map<String, byte[]> entries, final Manifest manifest) throws IOException {
        Path jarPath = tempDir.resolve(fileName);
        try (JarOutputStream out = null == manifest ? new JarOutputStream(Files.newOutputStream(jarPath)) : new JarOutputStream(Files.newOutputStream(jarPath), manifest)) {
            for (Entry<String, byte[]> each : entries.entrySet()) {
                out.putNextEntry(new JarEntry(each.getKey()));
                out.write(each.getValue());
                out.closeEntry();
            }
        }
        return new JarFile(jarPath.toFile());
    }
    
    private byte[] createClassBytes(final String className) {
        return new ByteBuddy().subclass(Object.class).name(className).make().getBytes();
    }
}
